using UnityEngine;
using System.Collections;
using System.Collections.Generic;

public class s_MotionTrail : MonoBehaviour
{
	List<GameObject> mTrails;			//List of gameobjects to be used as trails
	List<Position> mPositions;			//List of previous positions that trails use
	const int mNumOfTrails = 12;		//Number of trails
	int mActiveTrails = 0;				//Number of trails that are having their oppacity updated
	int mTrailChunk = 3;				//Number of trails per rewind speed (4 speeds * 3 = mNumofTrails)
	
	int mAddFrame = 4;					//Every nth frame that the trails are updated. Changes depending on rewind speed
	int mFrameCounter = 0;				//Actual frame counter
	
	float mMaxOpacity = 0.3f;			//Max transparancy
	float mFadeSpeed = 0.15f * 0.033f; 	//How fast it fades in and out. Didn't use delta time, because I want them to fade even when timescale=0
	float mFadeTol = 0.01f;				//For when checking if you have made it to where you are fading
	
	//Used to find out where trails should be
	public class Position
	{
		public float mX;
		public float mY;
		public float mXScale;
		public Texture mTexture;
		
		public void Set(float aX, float aY, float aXScale, Texture aTexture) { mX = aX; mY = aY; mXScale = aXScale; mTexture = aTexture; }
		
		public float GetX() 		{ return mX; }
		public float GetY() 		{ return mY; }
		public float GetXScale() 	{ return mXScale; }
		public Texture GetTexture() { return mTexture; }
	}
	
	void Start()
	{	
		s_TimeManager.Get.AddChangeFunction(RewindChange);
		s_TimeManager.Get.AddStartFunction(RewindChange);
		s_TimeManager.Get.AddEndFunction(RewindEnd);
		
		//Initialize lists
		mTrails = new List<GameObject>();
		foreach (Transform child in transform.parent.transform)
		{
			if (child == this.transform)
				continue;
			
			mTrails.Add(child.gameObject);
			
			Color col = child.renderer.material.color;
			child.renderer.material.color = new Color(col.r, col.g, col.b, 0);
		}
		
		mPositions = new List<Position>();
		for (int n = 0; n < mNumOfTrails; ++n)
		{
			Position temp = new Position();
			temp.Set(transform.position.x, transform.position.y, transform.lossyScale.x, renderer.material.mainTexture);
			mPositions.Add(temp);
		}
	}
	
	void LateUpdate()
	{	
		if (Time.timeScale == 0)
			return;
		
		//Updating counter
		if (mFrameCounter == mAddFrame)
		{
			mFrameCounter = 0;
			UpdateTrails();
		}
		else
			++mFrameCounter;
	}
	
	void RewindChange()
	{	
		//Depending on speed, changes number of active trails and how fast the positions are saved
		if (s_TimeManager.sRewindSpeed == 1 || s_TimeManager.sRewindSpeed == -1)
			mActiveTrails = 0;
		else if (s_TimeManager.sRewindSpeed == 2 || s_TimeManager.sRewindSpeed == -2)
			{ mActiveTrails = mTrailChunk*1; mAddFrame = 1; }
		else if (s_TimeManager.sRewindSpeed == 4 || s_TimeManager.sRewindSpeed == -4)
			{ mActiveTrails = mTrailChunk*2; mAddFrame = 0; }
		else if (s_TimeManager.sRewindSpeed == 8 || s_TimeManager.sRewindSpeed == -8)
			{ mActiveTrails = mTrailChunk*3; mAddFrame = 0; }
		
		
		mFrameCounter = mAddFrame;
		
		StopAllCoroutines();
		UpdateOpacity();
	}
	
	void UpdateOpacity()
	{
		//Set all previous ones to max
		int i = 0;
		for (; i < mActiveTrails; ++i)
			StartCoroutine(Fade(mTrails[i].renderer.material, mMaxOpacity));
		
		//Set next 3 from count to be fading
		for (int n = 1; i < mActiveTrails+mTrailChunk; ++i, ++n)
			StartCoroutine(Fade(mTrails[i].renderer.material, mMaxOpacity/n));
		
		//Set everything after to fade to black
		for (; i < mNumOfTrails; ++i)
			StartCoroutine(Fade(mTrails[i].renderer.material, 0));
	}
	
	void RewindEnd()
	{
		//Reset everything
		StopAllCoroutines();
		mAddFrame = 4;
		mActiveTrails = 0;
		
		//All fade to black
		foreach (GameObject go in mTrails)
			StartCoroutine(Fade(go.renderer.material, 0));
	}
	
	IEnumerator Fade(Material aMat, float aFadeTo)
	{
		Color col;
		
		while (aMat.color.a - aFadeTo > mFadeTol || aMat.color.a - aFadeTo < -mFadeTol)
		{
			col = aMat.color;
			
			//Add or take depening on where you are fading to
			if (col.a < aFadeTo)
				col.a += mFadeSpeed;
			else
				col.a -= mFadeSpeed;
			
			aMat.color = col;
			
			yield return null;
		}
		
		//Make sure to set it to exactly what it was supposed to be. 
		col = aMat.color;
		col.a = aFadeTo;
		aMat.color = col;
		
	}
	
	bool UpdatePositions()
	{	
		bool result = false;
		
		//Move last Position to first if you havent moved
		if(transform.position.x != mPositions[0].GetX()
		|| transform.position.y != mPositions[0].GetY())
		{
			result = true;
			Position last = mPositions[mNumOfTrails-1];
			mPositions.RemoveAt(mNumOfTrails-1);
			
			last.Set(transform.position.x, transform.position.y, transform.lossyScale.x, renderer.material.mainTexture);
			mPositions.Insert(0, last);
		}
		
		return result;
	}
	
	void UpdateTrails()
	{
		//Only update the trails if something moved
		if (!UpdatePositions())
			return;
		
		GameObject last;
		int index = 0;
		
		/*
		 	Basically, there are two sections of the mTrails, the active part and the inactive part:
		 		*The active part is at the front and is updated when rewinding.
		 		*The inactive part is updated at all times:
		 			*When rewinding, it is updating the portion that is not active, otherwise it updates everything
	 	*/
		
		if (s_TimeManager.sRewind)
			index = mActiveTrails+mTrailChunk;
		
		//Not active
		if (index < mNumOfTrails)
		{
			//Put last trail into the place after the active trails
			mTrails[mNumOfTrails-1].transform.position = new Vector3(mPositions[index].GetX(), mPositions[index].GetY(), transform.position.z);
			mTrails[mNumOfTrails-1].transform.localScale = new Vector3(mPositions[index].GetXScale(), transform.lossyScale.y, transform.lossyScale.z);
			mTrails[mNumOfTrails-1].renderer.material.mainTexture = mPositions[index].GetTexture();
			
			//Pop the last trail to the end of the active 
			last = mTrails[mNumOfTrails-1];
			mTrails.RemoveAt(mNumOfTrails-1);
			mTrails.Insert(index, last);
		}
		
		//Active
		if (s_TimeManager.sRewind)
		{
			//Put last of active trails into the first place
			mTrails[mActiveTrails+mTrailChunk-1].transform.position = new Vector3(mPositions[1].GetX(), mPositions[1].GetY(), transform.position.z);
			mTrails[mActiveTrails+mTrailChunk-1].transform.localScale = new Vector3(mPositions[1].GetXScale(), transform.lossyScale.y, transform.lossyScale.z);
			mTrails[mActiveTrails+mTrailChunk-1].renderer.material.mainTexture = mPositions[1].GetTexture();
			mTrails[mActiveTrails+mTrailChunk-1].renderer.material.color = mTrails[0].renderer.material.color;
			
			//Pop the last trail to the front
			last = mTrails[mActiveTrails+mTrailChunk-1];
			mTrails.RemoveAt(mActiveTrails+mTrailChunk-1);
			mTrails.Insert(0, last);
			
			//Re-update opacity
			StopAllCoroutines();
			UpdateOpacity();
		}
		
	}
}